<?php if ( ! defined( 'ABSPATH' ) ) { die( 'Direct access forbidden.' ); }
/**
 * Helper functions and classes with static methods for usage in theme
 */

/**
 * Register Lato Google font.
 *
 * @return string
 */
if ( ! function_exists( 'jws_theme_font_url' ) ) {
	function jws_theme_font_url() {
		$font_url = '';
		/*
		 * Translators: If there are characters in your language that are not supported
		 * by Lato, translate this to 'off'. Do not translate into your own language.
		 */
		if ( 'off' !== _x( 'on', 'Playfair Display font: on or off', 'thebrilliance' ) ) {
			$font_url = add_query_arg( 'family', urlencode( 'Poppins:300,300i,400,400i,500,500i,600,600i,700,700i' ),
				"//fonts.googleapis.com/css" );
		}

		return $font_url;
	}
}

/**
 * Getter function for Featured Content Plugin.
 *
 * @return array An array of WP_Post objects.
 */
if ( ! function_exists( 'jws_theme_get_featured_posts' ) ) {
	function jws_theme_get_featured_posts() {
		/**
		 * @param array|bool $posts Array of featured posts, otherwise false.
		 */
		return apply_filters( 'jws_theme_get_featured_posts', array() );
	}
}

/**
 * A helper conditional function that returns a boolean value.
 *
 * @return bool Whether there are featured posts.
 */
if ( ! function_exists( 'jws_theme_has_featured_posts' ) ) {
	function jws_theme_has_featured_posts() {
		return ! is_paged() && (bool) jws_theme_get_featured_posts();
	}
}

if ( ! function_exists( 'jws_theme_the_attached_image' ) ) : /**
 * Print the attached image with a link to the next attached image.
 */ {
	function jws_theme_the_attached_image() {
		$post = get_post();
		/**
		 * Filter the default attachment size.
		 *
		 * @param array $dimensions {
		 *     An array of height and width dimensions.
		 *
		 * @type int $height Height of the image in pixels. Default 810.
		 * @type int $width Width of the image in pixels. Default 810.
		 * }
		 */
		$attachment_size     = apply_filters( 'jws_theme_attachment_size', array( 810, 810 ) );
		$next_attachment_url = wp_get_attachment_url();

		/*
		 * Grab the IDs of all the image attachments in a gallery so we can get the URL
		 * of the next adjacent image in a gallery, or the first image (if we're
		 * looking at the last image in a gallery), or, in a gallery of one, just the
		 * link to that image file.
		 */
		$attachment_ids = get_posts( array(
			'post_parent'    => $post->post_parent,
			'fields'         => 'ids',
			'numberposts'    => - 1,
			'post_status'    => 'inherit',
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
			'order'          => 'ASC',
			'orderby'        => 'menu_order ID',
		) );

		// If there is more than 1 attachment in a gallery...
		if ( count( $attachment_ids ) > 1 ) {
			foreach ( $attachment_ids as $attachment_id ) {
				if ( $attachment_id == $post->ID ) {
					$next_id = current( $attachment_ids );
					break;
				}
			}

			// get the URL of the next image attachment...
			if ( $next_id ) {
				$next_attachment_url = get_attachment_link( $next_id );
			} // or get the URL of the first image attachment.
			else {
				$next_attachment_url = get_attachment_link( array_shift( $attachment_ids ) );
			}
		}

		printf( '<a href="%1$s" rel="attachment">%2$s</a>',
			esc_url( $next_attachment_url ),
			wp_get_attachment_image( $post->ID, $attachment_size )
		);
	}
}
endif;

if ( ! function_exists( 'jws_theme_list_authors' ) ) : /**
 * Print a list of all site contributors who published at least one post.
 */ {
	function jws_theme_list_authors() {
		$contributor_ids = get_users( array(
			'fields'  => 'ID',
			'orderby' => 'post_count',
			'order'   => 'DESC',
			'who'     => 'authors',
		) );

		foreach ( $contributor_ids as $contributor_id ) :
			$post_count = count_user_posts( $contributor_id );

			// Move on if user has not published a post (yet).
			if ( ! $post_count ) {
				continue;
			}
			?>

			<div class="contributor">
				<div class="contributor-info">
					<div class="contributor-avatar"><?php echo get_avatar( $contributor_id, 132 ); ?></div>
					<div class="contributor-summary">
						<h2 class="contributor-name"><?php echo get_the_author_meta( 'display_name',
								$contributor_id ); ?></h2>

						<p class="contributor-bio">
							<?php echo get_the_author_meta( 'description', $contributor_id ); ?>
						</p>
						<a class="button contributor-posts-link"
						   href="<?php echo esc_url( get_author_posts_url( $contributor_id ) ); ?>">
							<?php printf( _n( '%d Article', '%d Articles', $post_count, 'thebrilliance' ), $post_count ); ?>
						</a>
					</div>
					<!-- .contributor-summary -->
				</div>
				<!-- .contributor-info -->
			</div><!-- .contributor -->

		<?php
		endforeach;
	}
}
endif;

/**
 * Custom template tags
 */
{
	if ( ! function_exists( 'jws_paginate_links2' ) ) : 
		function jws_paginate_links2() {

			$paged = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;
			if ( function_exists( 'fw_get_db_settings_option' ) ){
				$post_setting = fw_get_db_settings_option('posts_settings');
			}else{
				$post_setting = array(
					'posts_pagination' => 'numeric',
				);
			}

			if($post_setting['posts_pagination'] == 'loadmore'){
				$encoded_atts = json_encode( $post_setting );
				$loadmore = '<nav class="navigation paging-navigation" role="navigation">
					<a href="#" class="blog-loadmore jws-load-more" data-atts="" data-paged="'.esc_attr($paged).'">'.esc_html__('Load More','thebrilliance').'</a>
					<p class="posts-loaded">'.esc_html__('All items Loaded','thebrilliance').'</p>
				</nav>';
				return $loadmore;
			}else{
			    $args = array( 
			        'prev_next'    => false,
			    );
			    $number_link = paginate_links( $args );
			    $previous 	= get_previous_posts_link( '<span class="prev">'.esc_html__('Previous','thebrilliance').'</span>' );
			    $next 		= get_next_posts_link( '<span class="next">'.esc_html__('Next','thebrilliance').'</span>'); 
				$link 		= '<nav class="navigation paging-navigation" role="navigation"><div class="pagination loop-pagination">'.$previous . $next . $number_link.'</div></nav>';
			    return $link;
		    }

		}
	endif;

	if ( ! function_exists( 'jws_theme_paging_nav' ) ) : /**
	 * Display navigation to next/previous set of posts when applicable.
	 */ {
		function jws_theme_paging_nav( $wp_query = null ) {

			if ( ! $wp_query ) {
				$wp_query = $GLOBALS['wp_query'];
			}

			// Don't print empty markup if there's only one page.

			if ( $wp_query->max_num_pages < 2 ) {
				return;
			}

			$paged        = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;
			$pagenum_link = html_entity_decode( get_pagenum_link() );
			$query_args   = array();
			$url_parts    = explode( '?', $pagenum_link );

			if ( isset( $url_parts[1] ) ) {
				wp_parse_str( $url_parts[1], $query_args );
			}

			$pagenum_link = remove_query_arg( array_keys( $query_args ), $pagenum_link );
			$pagenum_link = trailingslashit( $pagenum_link ) . '%_%';

			$format = $GLOBALS['wp_rewrite']->using_index_permalinks() && ! strpos( $pagenum_link,
				'index.php' ) ? 'index.php/' : '';
			$format .= $GLOBALS['wp_rewrite']->using_permalinks() ? user_trailingslashit( 'page/%#%',
				'paged' ) : '?paged=%#%';

			// Set up paginated links.
			$links = paginate_links( array(
				'base'      => $pagenum_link,
				'format'    => $format,
				'total'     => $wp_query->max_num_pages,
				'current'   => $paged,
				'mid_size'  => 1,
				'add_args'  => array_map( 'urlencode', $query_args ),
				'prev_text' =>esc_html__( '&larr; Previous', 'thebrilliance' ),
				'next_text' =>esc_html__( 'Next &rarr;', 'thebrilliance' ),
			) );

			if ( $links ) :

				?>
				<nav class="navigation paging-navigation" role="navigation">

					<div class="pagination loop-pagination">
						<?php echo wp_kses_post($links); ?>
					</div>
					<!-- .pagination -->
				</nav><!-- .navigation -->
			<?php
			endif;
		}
	}
	endif;

	if ( ! function_exists( 'jws_theme_post_nav' ) ) : /**
	 * Display navigation to next/previous post when applicable.
	 */ {
		function jws_theme_post_nav() {
			// Don't print empty markup if there's nowhere to navigate.
			$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '',
				true );
			$next     = get_adjacent_post( false, '', false );

			if ( ! $next && ! $previous ) {
				return;
			}

			?>
			<nav class="navigation post-navigation" role="navigation">

				<div class="nav-links">
					<?php
					if ( is_attachment() ) :
						previous_post_link( '%link',
							__( '<span class="meta-nav">Published In</span>', 'thebrilliance' ) );
					else :
						previous_post_link( '%link',
							__( '<span class="meta-nav"><span class="lnr lnr-chevron-left"></span> Previous Post</span>', 'thebrilliance' ) );
						next_post_link( '%link',wp_kses_post( '<span class="meta-nav">Next Post <span class="lnr lnr-chevron-right"></span></span>', 'thebrilliance' ) );
					endif;
					?>
				</div>
				<!-- .nav-links -->
			</nav><!-- .navigation -->
		<?php
		}
	}
	endif;

	if ( ! function_exists( 'jws_theme_event_nav' ) ) : /**
	 * Display navigation to next/previous post when applicable.
	 */ {
		function jws_theme_event_nav() {
			// Don't print empty markup if there's nowhere to navigate.
			$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '',
				true );
			$next     = get_adjacent_post( false, '', false );

			if ( ! $next && ! $previous ) {
				return;
			}

			?>
			<nav class="navigation post-navigation" role="navigation">
				<h1 class="screen-reader-text"><?php esc_html_e( 'Event navigation', 'thebrilliance' ); ?></h1>

				<div class="nav-links">
					<?php
					if ( is_attachment() ) :
						previous_post_link( '%link',
							__( '<span class="meta-nav">Published In</span>', 'thebrilliance' ) );
					else :
						previous_post_link( '%link',
							__( '<span class="meta-nav"><span class="lnr lnr-chevron-left"></span> Previous Event</span>', 'thebrilliance' ) );
						next_post_link( '%link',esc_html__( '<span class="meta-nav">Next Event <span class="lnr lnr-chevron-right"></span></span>', 'thebrilliance' ) );
					endif;
					?>
				</div>
				<!-- .nav-links -->
			</nav><!-- .navigation -->
		<?php
		}
	}
	endif;

	if ( ! function_exists( 'jws_theme_posted_on' ) ) : /**
	 * Print HTML with meta information for the current post-date/time and author.
	 */ {
		function jws_theme_posted_on() {
			if ( is_sticky() && is_home() && ! is_paged() ) {
				echo '<span class="featured-post">' .esc_html__( 'Sticky', 'thebrilliance' ) . '</span>';
			}

			// Set up and print post meta information.
			printf( '<span class="entry-date"><a href="%1$s" rel="bookmark"><time class="entry-date" datetime="%2$s">%3$s</time></a></span> <span class="byline"><span class="author vcard"><a class="url fn n" href="%4$s" rel="author">%5$s</a></span></span>',
				esc_url( get_permalink() ),
				esc_attr( get_the_date( 'c' ) ),
				esc_html( get_the_date() ),
				esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
				get_the_author()
			);
		}
	}
	endif;

	/**
	 * Find out if blog has more than one category.
	 *
	 * @return boolean true if blog has more than 1 category
	 */
	if ( ! function_exists( 'jws_theme_categorized_blog' ) ) {

		function jws_theme_categorized_blog() {
			if ( false === ( $all_the_cool_cats = get_transient( 'jws_theme_category_count' ) ) ) {
				// Create an array of all the categories that are attached to posts
				$all_the_cool_cats = get_categories( array(
					'hide_empty' => 1,
				) );

				// Count the number of categories that are attached to the posts
				$all_the_cool_cats = count( $all_the_cool_cats );

				set_transient( 'jws_theme_category_count', $all_the_cool_cats );
			}

			if ( 1 !== (int) $all_the_cool_cats ) {
				// This blog has more than 1 category so jws_theme_categorized_blog should return true
				return true;
			} else {
				// This blog has only 1 category so jws_theme_categorized_blog should return false
				return false;
			}
		}
	}

	/**
	 * Display an optional post thumbnail.
	 *
	 * Wraps the post thumbnail in an anchor element on index
	 * views, or a div element when on single views.
	 */
	if ( ! function_exists( 'jws_theme_post_thumbnail' ) ) {
		function jws_theme_post_thumbnail() {
			if ( post_password_required() || is_attachment() || ! has_post_thumbnail() ) {
				return;
			}

			$current_position = false;
			if (function_exists('fw_ext_sidebars_get_current_position')) {
				$current_position = fw_ext_sidebars_get_current_position();
			}



			if ( is_singular() ) :
				?>

				<div class="post-thumbnail">
					<?php
					if ( ( in_array( $current_position,
							array( 'full', 'left' ) ) || is_page_template( 'page-templates/full-width.php' )
						|| empty($current_position)
					)
					) {
						the_post_thumbnail( 'full' );
					} else {
						the_post_thumbnail();
					}
					?>
				</div>

			<?php else : ?>

				<a class="post-thumbnail" shref="<?php the_permalink(); ?>">
					<?php
					if ( ( in_array( $current_position,
							array( 'full', 'left' ) ) || is_page_template( 'page-templates/full-width.php' ) )
							|| empty($current_position)
					) {
						the_post_thumbnail( 'full' );
					} else {
						the_post_thumbnail();
					}
					?>
				</a>

			<?php endif; // End is_singular()
		}
	}
}

if ( ! function_exists( 'jws_list_pages' ) ){
	/**
	 * get an array of all pages
	 */
	function jws_list_pages() {
		$pages = get_pages();
		$result = array();
		$result[0] = esc_html__('None', 'thebrilliance');
		foreach ( $pages as $page ) {
			$result[ $page->ID ] = $page->post_title;
		}
		return $result;
	}
}

if ( ! function_exists( 'jws_list_footers' ) ){
	/**
	 * get an array of all pages
	 */
	function jws_list_footers() {
		$pages = get_posts([
		  'post_type' => 'footer',
		  'post_status' => 'publish',
		  'numberposts' => -1
		]);

		$result = array();
		$result[0] = esc_html__('None', 'thebrilliance');
		$result['footer-widget'] = esc_html__('Footer Widget', 'thebrilliance');
		foreach ( $pages as $page ) {
			$result[ $page->ID ] = $page->post_title;
		}
		return $result;
	}
}

if ( ! function_exists( 'jws_footer' ) ){
	/**
	 * Display theme footer
	 */
	function jws_footer() {

		if (defined( 'FW' )) {
			$footer_shortcode = jws_get_option_properly('footer_settings','enable_footer_meta','footer_shortcode');
			$shortcode_before_footer = jws_get_option_properly('footer_settings','enable_footer_meta','shortcode_before_footer');
		}else{
			$shortcode_before_footer = '';
			$footer_shortcode = array();
		}

		if ( class_exists( 'LP_Page_Controller' ) && function_exists('jws_dis_filter') ) {
			jws_dis_filter( 'the_content',  array(LP_Page_Controller::instance(), 'single_content') ,10000);
		}
		
		if ( class_exists( 'EM_Event_Post' ) && function_exists('jws_dis_filter') ) {
			 jws_dis_filter('the_content', array('EM_Event_Post','the_content'));
		}
        if (function_exists('jws_dis_filter2') ) {
		     jws_dis_filter2('the_content', 'em_content');
        }

		if ($shortcode_before_footer != ''){
			echo( apply_filters('the_content', $shortcode_before_footer,10001));
		}
		if ($footer_shortcode == '0') return false;
		?>
		<footer id="footer-default">
			<div class="main-footer">
				<?php
				if($footer_shortcode == 'footer-widget'){
					if (is_active_sidebar( 'footer-widget' )) {
						dynamic_sidebar( 'footer-widget' );
					}
				}else{
					echo jws_theme_content_visual($footer_shortcode);
				}
				?>
			</div>

			<?php 

			$copyright = esc_html__('Copyright &copy; thebrilliance 2019. All Right Reserved', 'thebrilliance' );
			if ( function_exists( 'fw_get_db_settings_option' ) && $footer_shortcode == 'footer-widget') :
				$jws_footer_settings = fw_get_db_settings_option('footer_settings');
				if ($jws_footer_settings['copyright'] != ''): 
					$copyright = $jws_footer_settings['copyright'];
				endif;
			endif;

			if (empty($footer_shortcode)):
			?>
				<div class="bottom-footer">
					<div class="container">
						<div class="row">
							<div class="col-md-12">
								<?php echo wp_kses_post($copyright); ?>
							</div>
						</div>
					</div>
				</div>
			<?php 
			endif;
			?>

		</footer>
	<?php }
}

if ( ! function_exists( 'jws_theme_header' ) ) {
    function jws_theme_header() {
        ob_start();
	    
	    $header_type_picker = jws_get_option_properly('header_settings','enable_header_meta','header_type_picker');
	    if(isset($header_type_picker['header_type'])){
	    	$layout = $header_type_picker['header_type'];
	    }else{
	    	$layout = "default";
	    }

        get_template_part( 'framework-theme/headers/header', $layout );
        $output = ob_get_clean();
        echo apply_filters( 'jws_theme_header', $output );
    }
}

if ( ! function_exists( 'jws_   ' ) ){
	/**
	 * Display theme logo
	 *
	 * @param boolean $wrap
	 */
	function jws_logo( $wrap = false ) {
		$jws_logo_settings['logo']['selected_value']   = 'text';
		$jws_logo_settings['logo']['text']['title']    = get_bloginfo( 'name' );
		$jws_logo_settings['logo']['text']['subtitle'] = '';

		if ( defined( 'FW' ) ) {
			$jws_logo_settings = jws_get_option_properly('','enable_general_meta','logo_settings');
		}
		?>
		<div class="jws-wrap-logo">
			<?php if ($wrap): ?>
				<div class="container">
			<?php endif; ?>

				<?php if ( $jws_logo_settings['logo']['selected_value'] == 'image' ) :
					$image_logo = $jws_logo_settings['logo']['image']['image_logo'];
					if ( ! empty( $image_logo ) ) : ?>
						<a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="site-logo">
                            <?php if (substr($image_logo['url'], -3, 3) == 'svg') echo '<span class="nxl_logo_svg">'; ?>
							<img src="<?php echo esc_url($image_logo['url']); ?>" alt="<?php echo esc_attr( bloginfo('name') ) ?>" />
                            <?php if (substr($image_logo['url'], -3, 3) == 'svg') echo '</span>'; ?>
						</a>
					<?php  endif;
				else :
					if ( $jws_logo_settings['logo']['text']['title'] != '' ) : ?>
						<a href="<?php echo esc_url( home_url('/') ); ?>" class="site-logo">
							<strong class="site-title"><?php echo wp_kses_post($jws_logo_settings['logo']['text']['title']); ?></strong>
						</a>
					<?php endif;
				endif; ?>

			<?php if ($wrap): ?>
				</div>
			<?php endif; ?>
		</div>
	<?php }
}


if ( ! function_exists( 'jws_logo_mobile' ) ){
	function jws_logo_mobile( $wrap = false ) {
		if ( defined( 'FW' ) ) {
			$jws_logo_settings = fw_get_db_settings_option( 'logo_mobile' );
		}
		?>
		<?php if (isset($jws_logo_settings['url'])) : ?>
			<div class="jws-wrap-logo">
				<a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="site-logo">
                    <?php if (substr($jws_logo_settings['url'], -3, 3) == 'svg') echo '<span class="nxl_logo_svg">'; ?>
					<img src="<?php echo esc_url($jws_logo_settings['url']); ?>" alt="<?php echo esc_attr( bloginfo('name') ) ?>" />
                    <?php if (substr($jws_logo_settings['url'], -3, 3) == 'svg') echo '</span>'; ?>
				</a>
			</div>
		<?php endif; ?>



		
	<?php }
}


// font variant
if ( ! function_exists( 'jws_convert_font_variant' ) ){
	function jws_convert_font_variant($font){
	    if($font == '')
	    	$font = '400';

	    $arr_variant = array(
	        '100' => '100_normal',
	        '100italic' => '100_italic',
	        '200' => '200_normal',
	        '200italic' => '200_italic',
	        '300' => '300_normal',
	        '300italic' => '300_italic',
	        '400' => '400_normal',
	        '400italic' => '400_italic',
	        'italic' => '400_italic',
	        'regular' => '400_normal',
	        '500' => '500_normal',
	        '500italic' => '500_italic',
	        '600' => '600_normal',
	        '600italic' => '600_italic',
	        '700' => '700_normal',
	        '700italic' => '700_italic',
	        '800' => '800_normal',
	        '800italic' => '800_italic',
	        '900' => '900_normal',
	        '900italic' => '900_italic',
	        'inherit' => 'inherit_inherit'
	    );
	    $new_font = explode("_",$arr_variant[$font]);
	    $arr_font['weight'] = $new_font[0];
	    $arr_font['style'] = $new_font[1];

	    return $arr_font;
	}
}

if(!function_exists('jws_theme_content_visual')){
	function jws_theme_content_visual($post_id='') {
		if(!$post_id) return false;
		$page_data = get_post( $post_id );
		$content = '';
		if ($page_data) {	
			$content .= apply_filters( 'the_content', $page_data->post_content );
			return $content;
		}
		else return false;	
	}
}

if(!function_exists('jws_get_typo_style')){
	function jws_get_typo_style($arr_typo ,$key ) {
		$jws_settings_option      	= fw_get_db_settings_option();
		$jws_color_settings       	= $jws_settings_option['color_settings'];
		$font_variant 	= jws_convert_font_variant($arr_typo['style']);
		$arr_typo_new = array(
			$key.'_font_family' => $arr_typo['family'],
			$key.'_font_weight' => $font_variant['weight'],
			$key.'_font_size' => $arr_typo['size'].'px',
			$key.'_text_color' => $arr_typo['color'],
		);
		return $arr_typo_new;
	}
}


if(!function_exists('jws_get_option_default')){
	function jws_get_option_default($key) {

	}
}

if ( ! function_exists( 'jws_get_socials' ) ) {
	/**
	 * Display socials buttons
	 *
	 * @param string $class
	 */
	function jws_get_socials( $class ) {
		$header_settings = fw_get_db_settings_option('header_settings'); 
		if ($header_settings['enable_top_bar']["yes"]['enable_header_socials']['selected_value'] == 'no') {
			return false;
		}

		$socials = fw_get_db_settings_option( 'socials' );



		if ( ! empty( $socials ) ) {
			$socials_html = '';
			// parse all socials
			foreach ( $socials as $social ) {
				$icon = '';
				if ( $social['social_type']['social-type'] == 'icon-social' ) {
					// get icon class
					if ( ! empty( $social['social_type']['icon-social']['icon_class'] ) ) {
						$icon .= '<i class="' . $social['social_type']['icon-social']['icon_class'] . '"></i>';
					}
				} else {
					// get uploaded icon
					if ( ! empty( $social['social_type']['upload-icon']['upload-social-icon'] ) ) {
						$icon .= '<img src="' . $social['social_type']['upload-icon']['upload-social-icon']['url'] . '" alt="'.esc_attr__('image','thebrilliance').'" />';
					}
				}

				// get social link
				$link = esc_url($social['social-link']);
				$title = '<div class="social-wrap-name"><div class="social-inner-name"><span>'.esc_html($social['social_name']).'</span></div></div>';

				$socials_html .= '<a target="_blank" href="' . $link . '">' . $icon .$title. '</a>';
			}

			// return socials html
			return '<div class="' . esc_attr($class) . '">' . $socials_html . '</div>';
		}
	}
}

add_action( 'wp_footer', 'jws_html_photoswipe' );

if( ! function_exists( 'jws_html_photoswipe' ) ) {
    function jws_html_photoswipe() { ?>

    <!-- Root element of PhotoSwipe. Must have class pswp. -->
    <div class="pswp" tabindex="-1" role="dialog" aria-hidden="true">

        <!-- Background of PhotoSwipe. 
             It's a separate element as animating opacity is faster than rgba(). -->
        <div class="pswp__bg"></div>

        <!-- Slides wrapper with overflow:hidden. -->
        <div class="pswp__scroll-wrap">

            <!-- Container that holds slides. 
                PhotoSwipe keeps only 3 of them in the DOM to save memory.
                Don't modify these 3 pswp__item elements, data is added later on. -->
            <div class="pswp__container">
                <div class="pswp__item"></div>
                <div class="pswp__item"></div>
                <div class="pswp__item"></div>
            </div>

            <!-- Default (PhotoSwipeUI_Default) interface on top of sliding area. Can be changed. -->
            <div class="pswp__ui pswp__ui--hidden">

                <div class="pswp__top-bar">

                    <!--  Controls are self-explanatory. Order can be changed. -->

                    <div class="pswp__counter"></div>

                    <button class="pswp__button pswp__button--close" title="<?php echo esc_attr__('Close (Esc)','thebrilliance'); ?>"></button>

                    <button class="pswp__button pswp__button--share" title="<?php echo esc_attr__('Share','thebrilliance'); ?>"></button>

                    <button class="pswp__button pswp__button--fs" title="<?php echo esc_attr__('Toggle fullscreen','thebrilliance'); ?>"></button>

                    <button class="pswp__button pswp__button--zoom" title="<?php echo esc_attr__('Zoom in/out','thebrilliance'); ?>"></button>

                    <div class="pswp__preloader">
                        <div class="pswp__preloader__icn">
                            <div class="pswp__preloader__cut">
                                <div class="pswp__preloader__donut"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="pswp__share-modal pswp__share-modal--hidden pswp__single-tap">
                    <div class="pswp__share-tooltip"></div> 
                </div>

                <button class="pswp__button pswp__button--arrow--left" title="<?php echo esc_attr__('Previous (arrow left)','thebrilliance'); ?>">
                </button>

                <button class="pswp__button pswp__button--arrow--right" title="<?php echo esc_attr__('Next (arrow right)','thebrilliance'); ?>">
                </button>

                <div class="pswp__caption">
                    <div class="pswp__caption__center"></div>
                </div>

            </div>

        </div>

    </div>

<?php
    }
}


if( ! function_exists( 'jws_additional_content' ) ) {
	function jws_additional_content() { 
		ob_start();
		?>
			<div class="share-link">
				<!-- facebook -->
				<a class="link-item fb" href="#">
					<div class="icon">
						<i class="fa fa-facebook"></i>
					</div>
					<div class="link-label">
						<span><?php esc_html_e('Share Post','thebrilliance'); ?></span>
					</div>
				</a>

				<!-- tweet -->
				<a class="link-item tw" href="#">
					<div class="icon">
						<i class="fa fa-twitter"></i>
					</div>
					<div class="link-label">
						<span><?php esc_html_e('Tweet Post','thebrilliance'); ?></span>
					</div>
				</a>

				<!-- gmail -->
				<a class="link-item gg" href="#">
					<div class="icon">
						<i class="fa fa-google-plus"></i>
					</div>
					<div class="link-label">
						<span><?php esc_html_e('Share Post','thebrilliance'); ?></span>
					</div>
				</a>


			</div>

		<?php
		return  ob_get_clean();
	}
}


if( ! function_exists( 'jws_after_content' ) ) {
	function jws_after_content() { 
		if(!has_tag())
			return false; 
		?>
		
			<?php

			if(function_exists('fw_get_db_settings_option')){
				$posts_settings = fw_get_db_settings_option('posts_settings');
			}else{
				$posts_settings = array(
					'tags_and_socials' => 'yes'
				);
			}

			if(isset($posts_settings['tags_and_socials']) && $posts_settings['tags_and_socials'] == 'yes'){
				echo '<div class="after-post">';
				the_tags( '<div class="tags-wrap"><span class="title">'.esc_html__('Tags: ','thebrilliance').'</span><span class="tag-links">', '', '</span></div>' ); 
				echo '</div>';
			} ?>
	<?php
	}
}

/*Author*/
if ( ! function_exists( 'jwstheme_author_render' ) ) {
	function jwstheme_author_render() {
		ob_start();
		if (get_the_author_meta('description') == '')
		return false;
		?>
		<div class="jws-about-author">
			<div class="jws-author-avatar"><?php echo get_avatar( get_the_author_meta( 'ID' ), 170 ); ?></div>
			<div class="jws-author-info">
				<div class="jws-name">
					<span><?php esc_html_e('Posted By','thebrilliance'); ?></span>
					<h5><?php the_author(); ?></h5>
				</div>
				<p class="description"><?php the_author_meta('description'); ?></p>
                <div class="icon-author">
                    <?php 
                    if(!empty(get_the_author_meta('facebook')))
                        printf( '<a href="%1$s"><i class="fa fa-facebook"></i></a>',esc_url(get_the_author_meta('facebook'))); 

                    if(!empty(get_the_author_meta('twitter')))
                        printf( '<a href="%1$s"><i class="fa fa-twitter"></i></a>',esc_url(get_the_author_meta('twitter')));

                    if(!empty(get_the_author_meta('gplus')))
                        printf( '<a href="%1$s"><i class="fa fa-google-plus"></i></a>',esc_url(get_the_author_meta('gplus')));

                    if(!empty(get_the_author_meta('linkedin')))
                        printf( '<a href="%1$s"><i class="fa fa-linkedin"></i></a>',esc_url(get_the_author_meta('linkedin')));
                    ?>
                </div>
			</div>
		</div>
		<?php
		return  ob_get_clean();
	} 
}

/**
 * Render related post based on post tags.
 *
 * @since 1.0.0
 */
if ( ! function_exists( 'jws_related_post' ) ) {
    function jws_related_post($post_setting) {
        global $post;
        // Get page options
        $options = get_post_meta( get_the_ID(), '_custom_post_options', true );
        wp_enqueue_script( 'slick');
        // Get post's tags
        $tags = wp_get_post_tags( $post->ID );
        $categories = wp_get_post_categories($post->ID);
        if ( $tags || $categories) {
            // Get id for all tags
            $tag_ids = array();

            foreach ( $tags as $tag ) {
                $tag_ids[] = $tag->term_id;
            }


            // Build arguments to query for related posts
            $args = array(
            	'category__in' 		  => $categories,
                'tag__in'             => $tag_ids,
                'post__not_in'        => array( $post->ID ),
                'posts_per_page'      => (int)$post_setting['related_columns'],
                'ignore_sticky_posts' => 1,
                'orderby'             => 'rand',
            );
            $arr_col = array(
                2   => 'col-md-6 col-sm-6 col-xs-12',
                3   => 'col-md-4 col-sm-6 col-xs-12',
                4   => 'col-md-3 col-sm-6 col-xs-12'
            );

            // Get related post
            $related = new wp_query( $args );


            if(!$related->have_posts())
                return false;
            ?>
            <div class="post-related">
            	<div class="container">
            		<div class="row">
            			
            		<div class="col-md-12">
		                <div class="wrap-title">
		                    <h3 class="related-title"><?php esc_html_e('Related News','thebrilliance'); ?></h3>
		                    <div class="sub-title"><span><?php esc_html_e('Journals','thebrilliance'); ?></span></div>
		                </div>

		                <div class="jws-related-post jws-<?php echo esc_attr($post_setting['related_style']); ?>">
		                	<div class="row">
		                		
		                    <?php
		                    while ( $related->have_posts() ):
		                        $related->the_post();
		                        ?>
		                            
		                            <div class="<?php echo esc_attr($arr_col[(int)$post_setting['related_columns']]); ?>">
		                                <div class="post-item">
		                                    <div class="blog-image">
		                                    	
		                                        <?php 
		                                        the_post_thumbnail('jws-related-post-3col'); 
		                                        switch (get_post_format()) {
		                                        	case 'video':
		                                        		$icon = '<i class="icon-video-camera"></i>';
		                                        		break;

		                                        	case 'gallery':
		                                        		$icon = '<i class="icon-images"></i>';
		                                        		break;

		                                        	case 'image':
		                                        		$icon = '<i class="icon-camera"></i>';
		                                        		break;

		                                        	case 'quote':
		                                        		$icon = '<i class="icon-quotes-left"></i>';
		                                        		break;
		                                        	
		                                        	default:
		                                        		$icon = '<i class="icon-file-text2"></i>';
		                                        		break;
		                                        }
		                                        ?>
		                                        <?php if (has_post_thumbnail()) : ?>
											    	<div class="thumb-format"><?php echo wp_kses_post($icon); ?></div>
											    <?php endif; ?>
		                                        <div class="readmore"><a href="<?php the_permalink(); ?> "><?php esc_html_e('Learn More','thebrilliance'); ?></a></div>
		                                    </div>
		                                    <div class="content-blog">
		                                        <h5 class="title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h5>

													<div class="meta-post">
														<div class="author"><span class="icon-user"></span>
															<?php 
																$post_author_id = get_post_field( 'post_author', get_the_ID() );
																$author = get_userdata( $post_author_id );
																echo '<a href="'.get_author_posts_url($related->post_author).'">'.esc_attr($author->display_name).'</a>';
															?> 
														</div>
														<div class="date-post"><span class="icon-calendar2"></span><?php echo date_i18n( get_option( 'date_format' ), strtotime( $related->post_date ) ); ?></div>
													</div>
		                                    </div>
		                                </div>
		                            </div>
		                    <?php   
		                    endwhile;
		                    ?>

		                    </div>
		                </div> 
		            </div>

                	</div>
            	</div> 

            </div>
            <?php
            // Reset global query object
            wp_reset_postdata();
        }
    }
}


if( ! function_exists( 'jws_header_image' ) ) {
	/**
	 * Display header image for taxonomies and posts
	 */
	function jws_header_image(){
		if (!defined('FW')) {
			return;
		}
		if (is_page_template( 'page-templates/front-page.php' ) || is_front_page()) {
			return false;
		}

		global $post;
		$post_type = get_post_type( $post );

		if( is_search() ){
			// for search get options form general/posts
			$post_type = 'post';
		}
		
		$general_bc_and_oh = fw_get_db_settings_option( 'general_bc_and_oh', '' );
			
		$overlay_header_height = $content_position_style = $description = $jws_overlay_style = $title =  $taxonomy = $term_id = $header_image_overlap_style = $header_image_overlap_class = '';

			if (!is_404() && !is_search()) {
				if ($post) {
					$post_id = $post->ID;
				}else{
					$post_id = 0;
				}
				
				$image   = fw_get_db_post_option($post_id, 'general_bc_and_oh', '');
				if($image == ''){
					// if image from page is empty - get image from general theme settings
					$image = isset($general_bc_and_oh['overlay_header_image']['url']) ? $general_bc_and_oh['overlay_header_image']['url'] : array();
				}
			}

			if(is_page()) { 
			
                 $pages = get_queried_object_id();
                $title = get_the_title($pages);
              
              
			}elseif(is_home()){
				$title = esc_html__('Blogs','thebrilliance');
			}elseif(is_404()){
				$title = esc_html__('404 Not Found','thebrilliance');
			}elseif(is_search()){
				$title = sprintf(esc_html__( 'Search Results for: %s', 'thebrilliance' ), get_search_query() );
			}elseif(is_singular('post')){
				$title = esc_html__('Blog Single','thebrilliance');
			}elseif(is_post_type_archive( 'tribe_events' )){
                $title = esc_html__('Events', 'thebrilliance');
            }elseif(is_post_type_archive( 'lp_course')){
                $title = esc_html__('Courses', 'thebrilliance');
            }else{
				$title = get_the_title($post_id);
			}
			$jws_overlay_bg = $jws_opacity = '';
			// overlay
			$overlay_header_options = isset( $general_bc_and_oh['overlay_header_options'] ) ? $general_bc_and_oh['overlay_header_options'] : array();
			if ( isset( $overlay_header_options['header_overlay'] ) && $overlay_header_options['header_overlay'] == 'yes' ) {
				$jws_overlay_bg = $overlay_header_options['yes']['overlay_color'];
				$jws_opacity    = $overlay_header_options['yes']['overlay_opacity_image'] / 100;
			}

			// header height and header content position
			$overlay_header_height = isset($general_bc_and_oh['overlay_header_height']) ? $general_bc_and_oh['overlay_header_height'] : '';
			$posts_header_content_position = isset($general_bc_and_oh['posts_header_content_position']) ? $general_bc_and_oh['posts_header_content_position'] : '';
			$content_position_style = !empty($posts_header_content_position) ? 'style="top:'.$posts_header_content_position.'px;"' : '';
			if ( $overlay_header_height != 'auto' && $overlay_header_height != 'section-height-sm' && $overlay_header_height != 'section-height-md' && $overlay_header_height != 'section-height-lg' ) {
				$header_image_overlap_style .= ' padding: ' . (int) $overlay_header_height . 'px 0;';
				$overlay_header_height         = 'section-height-custom';
			} else {
				$overlay_header_height = $overlay_header_height;
			}

			// header image overlap
			$header_image_overlap = isset($general_bc_and_oh['header_image_overlap']) ? $general_bc_and_oh['header_image_overlap'] : '';
		

		if(!empty($image) || (isset($overlay_header_options['header_overlay']) && $overlay_header_options['header_overlay'] =='yes') || (isset($overlay_header_options['category_header_overlay']) && $overlay_header_options['category_header_overlay'] =='yes') ){ ?>
			<section class="overlay-header <?php echo esc_attr($overlay_header_height); ?>" style="background-image: url('<?php if( !empty( $image ) ) echo esc_url( $image ); ?>'); ">

				<div class="main-row-overlay" style="<?php
					if (isset($jws_overlay_bg) && !empty($jws_overlay_bg)){echo 'background-color:'.esc_attr($jws_overlay_bg).';';}
					if(isset($jws_opacity)){echo 'opacity: '.esc_attr($jws_opacity).';';} ?>"></div>

				<div class="container">
					<div class="row">
						<div class="col-md-12">
							<div class="heading heading-center" style="<?php echo esc_attr($header_image_overlap_style); ?>">
								<h1 class="special-title"><?php echo jws_translate($title); ?></h1>
								<?php if($description != '' ) : ?>
									<div class="special-subtitle"><?php echo jws_translate($description); ?></div>
								<?php endif; ?>

								<?php 
								if(is_page() || is_singular() || is_archive() || is_home() || is_single()){

									if ( function_exists( 'fw_get_db_settings_option' ) ){
										$general_bc_and_oh = fw_get_db_settings_option('general_bc_and_oh');
									}else{
										$general_bc_and_oh = array(
											'enabled_breadcrumb' => 'no',
											'breadcrumb_separator' => '/',
										);
									}

									if($general_bc_and_oh['enabled_breadcrumb'] == 'yes'){

										echo '<div class="wrap-breadcrumbs"><span class="icon-home4"></span>';
										if( function_exists('fw_ext_breadcrumbs') ) fw_ext_breadcrumbs($general_bc_and_oh['breadcrumb_separator']);
										echo '</div>';
									}
								}
								?>
							</div>
						</div>
					</div>
				</div>
			</section>
		<?php
		} 
	}
}

if ( ! function_exists( 'jws_post_meta' ) ) {
	function jws_post_meta() { 
		global $post;
		?>

		<div class="meta-post">
			<div class="author"><span class="lnr lnr-user"></span>
				<?php 
					$author = get_userdata( $post->post_author ); 
					echo '<a href="'.get_author_posts_url($post->post_author).'">'.esc_attr($author->display_name).'</a>';
				?> 
			</div>
			<div class="date-post"><span class="lnr lnr-clock"></span><?php echo date_i18n( get_option( 'date_format' ), strtotime( $post->post_date ) ); ?></div>
			<div class="num-comments"><span class="lnr lnr-menu-circle"></span><?php comments_number( esc_html__( 'No Comments', 'thebrilliance' ), esc_html__( '1 Comment', 'thebrilliance' ), esc_html__( '% Comments', 'thebrilliance' ) ); ?></div>
		</div>
		<?php
	}
}
if ( ! function_exists( 'jws_translate' ) ) {
	/**
	 * Return the content for translations plugins
	 *
	 * @param string $content
	 */
	function jws_translate( $content ) {
		$content = html_entity_decode( $content, ENT_QUOTES, 'UTF-8' );

		if ( function_exists( 'icl_object_id' ) && strpos( $content, 'wpml_translate' ) == true ) {
			$content = do_shortcode( $content );
		} elseif ( function_exists( 'qtrans_useCurrentLanguageIfNotFoundUseDefaultLanguage' ) ) {
			$content = qtrans_useCurrentLanguageIfNotFoundUseDefaultLanguage( $content );
		} elseif ( function_exists('ppqtrans_useCurrentLanguageIfNotFoundUseDefaultLanguage' ) ) {
			$content = ppqtrans_useCurrentLanguageIfNotFoundUseDefaultLanguage($content);
		}

		return $content;
	}
}

/* Filter body class */
add_filter( 'body_class', 'jws_body_class' );

if ( ! function_exists( 'jws_body_class' ) ) {
	function jws_body_class( $classes ) {
		global $sidebar;

		if(is_active_sidebar( $sidebar )){
			$classes[] = 'has-sidebar';
		}

		if ( function_exists( 'fw_get_db_settings_option' ) ){
			$container_site_type = jws_get_option_properly('','enable_general_meta','container_site_type');
			$classes[] = $container_site_type['selected'];

			$header_settings = fw_get_db_settings_option('header_settings');

			if($header_settings['enable_sticky_header'] == "fw-header-sticky"){
				$classes[] = 'enabled-sticky';
			}

			$enable_rtl = fw_get_db_settings_option('enable_rtl');
			if($enable_rtl == 'yes'){
				$classes[] = 'jws-rtl';
			}

			$post_option_meta = fw_get_db_post_option(get_the_ID(),'page_settings');
			if($post_option_meta['enable_fullpage'] == 'yes'){
				$classes[] = 'jws-fullpage';
			}
		}

	    return $classes;
	}
}

if ( ! function_exists( 'jws_go_to_top_button' ) ) {
	function jws_go_to_top_button() {
		if ( function_exists( 'fw_get_db_settings_option' ) ) : 
			if(fw_get_db_settings_option('enable_scroll_to') == 'no') return;
		endif;
	?>

		<a id="back-to-top">
	        <div>
		    	<i class="fa fa-angle-up"></i>
		    </div>
	    </a>
	<?php
	}
}

if ( ! function_exists( 'jws_is_page_url_excluded' ) ){
	/**
	 * Check if is page is from excluded pages
	 */
	function jws_is_page_url_excluded() {
		$exception_urls = array( 'wp-login.php', 'async-upload.php', '/plugins/', 'wp-admin/', 'upgrade.php', 'trackback/', 'feed/' );

		return false;
	}
}

if ( ! function_exists( 'jws_get_option_properly' ) ){
	function jws_get_option_properly($db_option, $page_option, $option_target) {
		if ( function_exists( 'fw_get_db_settings_option' ) ) :
			$option_db = fw_get_db_settings_option($db_option);
			$post_option_meta = fw_get_db_post_option(get_the_ID(),'page_settings');
			if (isset($post_option_meta[$page_option]["selected_value"]) && $post_option_meta[$page_option]["selected_value"] == 'yes' ) {
				if (isset($post_option_meta[$page_option]["yes"][$option_target])) {
					return $post_option_meta[$page_option]["yes"][$option_target];
				}else{
					return false;
				}
			}else{
				return $option_db[$option_target];
			}
		endif;
	}
}

if (!function_exists('jws_favicon')) {
    add_action('wp_head', 'jws_favicon');
    function jws_favicon(){

        $favicon = JWS_URI_PATH.'/images/favicon.ico';
    	if ( function_exists( 'fw_get_db_settings_option' ) ) :
        	$favicon_arr = fw_get_db_settings_option('favicon');
            if(isset($favicon_arr['url']) && !empty($favicon_arr['url']) ) {
                $favicon = $favicon_arr['url'];
            }
        endif;

        if (!function_exists('has_site_icon') || !has_site_icon()) {
            printf('<link rel="shortcut icon" href="' . esc_url($favicon) . '" />');
        }
    }
}

add_action( 'template_redirect', function(){
    ob_start( function( $buffer ){
        // Also works with other attributes...
        $buffer = str_replace( array( 'frameborder="0"', "" ), '', $buffer );
        return $buffer;
    });
});

//Create the rating interface.
add_action( 'comment_form_submit_button', 'jws_comment_rating_rating_field', 10, 2 );


function jws_comment_rating_rating_field ($submit_button, $args) {
	?>
	<label><?php esc_html_e('Your Rating','thebrilliance'); ?></label>
	<fieldset class="comments-rating">
		<span class="rating-container">
			<?php for ( $i = 5; $i >= 1; $i-- ) : $count = $i;?>
				<input type="radio" id="rating-<?php echo esc_attr( $i ); ?>" name="rating" value="<?php echo esc_attr( $i ); ?>" />
				<label for="rating-<?php echo esc_attr( $i ); ?>">
					<?php 
					for ( $count; $count >= 1; $count -- ) : ?>

					<i class="fa fa-star-o" aria-hidden="true"></i>

					<?php endfor; ?>
					
						
				</label>
			<?php endfor; ?>
		</span>
	</fieldset>
	<?php
	echo wp_kses($submit_button,array(
		'input' => array(
			'name' => array(),
			'type' => array(),
			'id' => array(),
			'class' => array(),
			'value' => array(),
		),
	));
}

//Save the rating submitted by the user.
add_action( 'comment_post', 'jws_comment_rating_save_comment_rating' );
function jws_comment_rating_save_comment_rating( $comment_id ) {
	if ( ( isset( $_POST['rating'] ) ) && ( '' !== $_POST['rating'] ) )
	$rating = intval( $_POST['rating'] );
	add_comment_meta( $comment_id, 'rating', $rating );
}

//Make the rating required.
add_filter( 'preprocess_comment', 'jws_comment_rating_require_rating' );
function jws_comment_rating_require_rating( $commentdata ) {
	if ( ! isset( $_POST['rating'] ) || 0 === intval( $_POST['rating'] ) ):
		if (is_singular('post')) {
			wp_die(esc_html__( 'Error: You did not add a rating. Hit the Back button on your Web browser and resubmit your comment with a rating.','thebrilliance' ) );
		}
	endif;

	return $commentdata;
}

//Display the rating on a submitted comment.
add_filter( 'comment_text', 'jws_comment_rating_display_rating');
function jws_comment_rating_display_rating( $comment_text ){

	if ( ($rating = get_comment_meta( get_comment_ID(), 'rating', true ) ) && is_singular('post') ) {
		$stars = '<p class="stars">';
		for ( $i = 1; $i <= $rating; $i++ ) {
			$stars .= '<i class="fa fa-star" aria-hidden="true"></i>';
		}
		$stars .= '</p>';
		$comment_text = $comment_text . $stars;
		return $comment_text;
	} else {
		return $comment_text;
	}
}

// Jws get post image
if ( ! function_exists( 'jws_get_post_thumbnail' ) ) {
    function jws_get_post_thumbnail( $size = 'medium', $attach_id = false ) {
        global $post, $jws_loop;
        if ( has_post_thumbnail() ) {
            if( function_exists( 'wpb_getImageBySize' ) ) {
                if( ! $attach_id ) $attach_id = get_post_thumbnail_id();
                if( ! empty( $jws_loop['img_size'] ) ) $size = $jws_loop['img_size'];
                $img = wpb_getImageBySize( array( 'attach_id' => $attach_id, 'thumb_size' => $size, 'class' => 'attachment-large wp-post-image' ) );
                $img = $img['thumbnail'];
            } else {
                $img = get_the_post_thumbnail( $post->ID, $size );
            }
            return $img;
        }
    }
}


if ( ! function_exists( 'jwstheme_paging_nav' ) ) {
    function jwstheme_paging_nav($query) {
        if ( $query->max_num_pages < 2 ) {
            return;
        }

        $paged        = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;
        $pagenum_link = html_entity_decode( get_pagenum_link() );
        $query_args   = array();
        $url_parts    = explode( '?', $pagenum_link );

        if ( isset( $url_parts[1] ) ) {
            wp_parse_str( $url_parts[1], $query_args );
        }

        $pagenum_link = remove_query_arg( array_keys( $query_args ), $pagenum_link );
        $pagenum_link = trailingslashit( $pagenum_link ) . '%_%';

        $format  = $GLOBALS['wp_rewrite']->using_index_permalinks() && ! strpos( $pagenum_link, 'index.php' ) ? 'index.php/' : '';
        $format .= $GLOBALS['wp_rewrite']->using_permalinks() ? user_trailingslashit( 'page/%#%', 'paged' ) : '?paged=%#%';

        global $pagination, $grid_columns, $limit;
		if ( function_exists( 'fw_get_db_settings_option' ) ){
			$course_settings = fw_get_db_settings_option('course_settings');
		}else{
			$course_settings = array(
				'archive_course_pagination' => 'numeric',
			);
		}

		$course_settings['archive_course_limit_posts'] = $limit;


		if(isset($_GET['layout'])){ 
			$course_settings['archive_course_layout'] = $_GET['layout']; 
		}else{
			$course_settings['archive_course_layout'] = $grid_columns;
			
		}

		$encoded_atts = json_encode( $course_settings );

        // Set up paginated links.
        $links = paginate_links( array(
                'base'     => $pagenum_link,
                'format'   => $format,
                'total'    => $query->max_num_pages,
                'current'  => $paged,
                'mid_size' => 1,
                'add_args' => array_map( 'urlencode', $query_args ),
                'prev_text' => '',
                'next_text' => '',
        ) );

        
        if ( $links ) { ?>
        <nav class="navigation paging-navigation" role="navigation">
            <?php if ($pagination == 'loadmore'): ?>
                <a href="#" class="course-loadmore jws-load-more" data-atts="<?php echo esc_attr($encoded_atts); ?>" data-paged="<?php echo esc_attr($paged); ?>"><?php esc_html_e('Load More', 'thebrilliance'); ?></a>
                <p  class="posts-loaded"><?php esc_html_e('All Posts Loaded.', 'thebrilliance'); ?></p>
            <?php elseif( $pagination == 'numeric' ): ?>
				<?php  
				$args = array( 
			        'prev_next'    => false,
			    );
			    $number_link = paginate_links( $args );
			    $previous 	= get_previous_posts_link( '<span class="prev">'.esc_html__('Previous','thebrilliance').'</span>' );
			    $next 		= get_next_posts_link('<span class="next">'.esc_html__('Next','thebrilliance').'</span>', $query->max_num_pages );
				$link 		= '<nav class="navigation paging-navigation" role="navigation"><div class="pagination loop-pagination">'.$previous . $links. $next .'</div></nav>';
			    echo wp_kses_post($link); 
			    ?>
            <?php endif ?>
        </nav>
        <?php
        }
    }
}

add_filter( 'woocommerce_output_related_products_args', 'jws_related_products_args' );

function jws_related_products_args( $args ) {
    $args['posts_per_page'] = 4; 
    return $args;
}
/**
* Enables the Thumbnail support for Organizer and Venue edit screens.
*/
function tribe_add_thumbnail_support_venue_organizer() {
add_post_type_support( 'tribe_organizer', 'thumbnail' );
add_post_type_support( 'tribe_venue', 'thumbnail' );
}
add_action( 'init', 'tribe_add_thumbnail_support_venue_organizer' );
/*
* Remove Ajax from The Events Calendar Pagination on Month, List, and Day Views
* @3.8
*
*/
add_action('wp_print_scripts', 'events_calendar_remove_scripts' , 10);
/*
* Remove Ajax from The Events Calendar Pagination on Month, List, and Day Views
* @3.8
*
*/
add_action('wp_print_scripts', 'events_calendar_remove_scripts' , 10);
function events_calendar_remove_scripts() {
 
        wp_dequeue_script( 'tribe-events-calendar');   
        wp_dequeue_script( 'tribe-events-list');       
        wp_dequeue_script( 'tribe-events-ajax-day');   
 
}
function add_file_types_to_uploads($file_types){
$new_filetypes = array();
$new_filetypes['svg'] = 'image/svg+xml';
$file_types = array_merge($file_types, $new_filetypes );
return $file_types;
}
add_action('upload_mimes', 'add_file_types_to_uploads');
if (!function_exists('thebrilliance_get_sidebars')) :
    function thebrilliance_get_sidebars($data = array())
    {
        global $wp_registered_sidebars;

        $result = array();
        foreach ($wp_registered_sidebars as $sidebar) {
            $result[$sidebar['id']] = $sidebar['name'];
        }

        //
        if (count($data) > 0) {
            $result = array_merge($data, $result);
        }

        return $result;
    }
endif;

if (!function_exists('thebrilliance_get_search_ajax')) :
    function thebrilliance_get_search_ajax()
    { ?>
        <div class="search-overlay" style="display: none;">
                <?php thebrilliance_get_search_ajax_inner(); ?>
    	</div>  
    <?php }
    add_action('wp_footer','thebrilliance_get_search_ajax');
endif;

if (!function_exists('thebrilliance_get_search_ajax_inner')) :
    function thebrilliance_get_search_ajax_inner()
    { ?>
            
    		<div class="jws-fullscreen-search-shader"></div>
    		<div class="jws-search-table">
    			<div class="jws-search-cell">
    				<div class="jws-search-inner">
    				
    					
    		
    								<div class="jws-search-form">
                        			
                        				<form role="search" method="get" class="searchform has-categories-dropdown search-style-default jws-ajax-search" action="<?php echo esc_url( home_url( '' ) ); ?>" data-thumbnail="1" data-price="1" data-post_type="course" data-count="20" >
                        					<input type="text" class="s" placeholder="<?php echo esc_attr_x( 'What do you want to learn?', 'placeholder', 'thebrilliance' ); ?>" value="<?php echo get_search_query(); ?>" name="s" />
                        					<input type="hidden" name="ref" value="<?php echo esc_attr('course'); ?>">
                        					<?php jws_show_categories_dropdown(); ?>
                        					<button type="submit" class="searchsubmit">
                        						<span class="icon-search32"></span>
                        					</button>
                        				</form>
                        			
                        				<div class="search-results-wrapper"><div class="jws-scroll"><div class="jws-search-results jws-scroll-content"></div></div><div class="jws-search-loader"></div></div>
                        				
                        			</div>
    								<i class="btn-close fa fa-times" aria-hidden="true"></i>
    
    				
    				</div>
    			</div>
    		</div>
    	  
    <?php }
  
endif;

if( ! function_exists( 'jws_show_categories_dropdown' ) ) {
	function jws_show_categories_dropdown() {
		$args = array( 
			'hide_empty' => 1,
			'parent' => 0
		);
		$terms = get_terms('course_category', $args);
		if( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
			?>
			<div class="search-by-category input-dropdown">
				<div class="input-dropdown-inner jws-scroll-content">
					<input type="hidden" name="course_category" value="0">
					<a href="#" data-val="0"><?php esc_html_e('Select category', 'thebrilliance'); ?></a>
					<div class="list-wrapper jws-scroll">
						<ul class="jws-scroll-content">
							<li style="display:none;"><a href="#" data-val="0"><?php esc_html_e('Select category', 'thebrilliance'); ?></a></li>
							<?php
						
								 foreach ( $terms as $term ) {
								    	?>
											<li><a href="#" data-val="<?php echo esc_attr( $term->slug ); ?>"><?php echo esc_attr( $term->name ); ?></a></li>
								    	<?php
								 }
							
							?>
						</ul>
					</div>
				</div>
			</div>
			<?php
		}
	}
}


if ( ! function_exists( 'jws_ajax_suggestions' ) ) {
	function jws_ajax_suggestions() {


		$query_args = array(
			'posts_per_page' => 5,
			'post_status'    => 'publish',
			'post_type'      => 'lp_course',
			'no_found_rows'  => 1,
		);




		if ( ! empty( $_REQUEST['query'] ) ) {
			$query_args['s'] = sanitize_text_field( $_REQUEST['query'] );
		}

		if ( ! empty( $_REQUEST['number'] ) ) {
			$query_args['posts_per_page'] = (int) $_REQUEST['number'];
		}
        
        if ( ! empty( $_REQUEST['course_category'] ) ) {
				$query_args['course_category'] = strip_tags( $_REQUEST['course_category'] );
		}

		$results = new WP_Query( apply_filters( 'jws_ajax_get_results', $query_args ) );

		$suggestions = array();

		if ( $results->have_posts() ) {

		  

			while ( $results->have_posts() ) {
				$results->the_post();
                $course = LP_Global::course();
		      	$price_html = $course->get_price_html();
                if (function_exists('wpb_getImageBySize')) {
    				$img = wpb_getImageBySize( array( 'attach_id' => get_post_thumbnail_id(get_the_ID()), 'thumb_size' => '300x300', 'class' => 'jws_course_image' ) );
    				$img = $img['thumbnail'];
                }else {
                    $img = get_the_post_thumbnail( get_the_ID(), 'thumbnail', array( 'class' => '' ) );
                }								
					$suggestions[] = array(
						'value' => get_the_title(),
						'permalink' => get_the_permalink(),
						'thumbnail' => wp_kses_post($img),
                        'price' => wp_kses_post($price_html),
					);
			
			}

			wp_reset_postdata();
		} else {
			$suggestions[] = array(
				'value' => esc_html__( 'No course found', 'thebrilliance' ),
				'no_found' => true,
				'permalink' => ''
			);
		}

		echo json_encode( array(
			'suggestions' => $suggestions
		) );

		die();
	}

	add_action( 'wp_ajax_jws_ajax_search', 'jws_ajax_suggestions', 10 );
	add_action( 'wp_ajax_nopriv_jws_ajax_search', 'jws_ajax_suggestions', 10 );
}